class MenuDisplay {
    constructor(config) {
        this.apiUrl = config.apiUrl;
        this.boardSlug = config.boardSlug;
        this.refreshInterval = config.refreshInterval || 30000;
        this.menuData = null;
        this.refreshTimer = null;
        this.isOnline = navigator.onLine;
        this.cacheKey = `menu_cache_${this.boardSlug}`;
        
        // DOM elements
        this.container = document.getElementById('menu-content');
        this.businessName = document.getElementById('business-name');
        this.currentTime = document.getElementById('current-time');
        this.lastUpdated = document.getElementById('last-updated');
        this.loading = document.getElementById('loading');
        
        // Bind methods
        this.fetchMenu = this.fetchMenu.bind(this);
        this.renderMenu = this.renderMenu.bind(this);
        this.updateClock = this.updateClock.bind(this);
        
        // Setup event listeners
        window.addEventListener('online', () => {
            this.isOnline = true;
            this.fetchMenu();
        });
        
        window.addEventListener('offline', () => {
            this.isOnline = false;
            console.log('Offline mode: Using cached data');
        });
    }
    
    init() {
        // Start clock
        this.updateClock();
        setInterval(this.updateClock, 1000);
        
        // Load cached data first (for offline capability)
        this.loadFromCache();
        
        // Fetch fresh data
        this.fetchMenu();
        
        // Setup auto-refresh
        this.startAutoRefresh();
    }
    
    updateClock() {
        const now = new Date();
        const hours = String(now.getHours()).padStart(2, '0');
        const minutes = String(now.getMinutes()).padStart(2, '0');
        const seconds = String(now.getSeconds()).padStart(2, '0');
        this.currentTime.textContent = `${hours}:${minutes}:${seconds}`;
    }
    
    async fetchMenu() {
        try {
            const response = await fetch(`${this.apiUrl}?board=${this.boardSlug}`);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const data = await response.json();
            
            if (data.success) {
                this.menuData = data;
                this.saveToCache(data);
                this.renderMenu(data);
                
                // Update refresh interval from server
                if (data.board.refreshInterval) {
                    this.updateRefreshInterval(data.board.refreshInterval * 1000);
                }
            } else {
                throw new Error(data.error || 'Failed to load menu');
            }
            
        } catch (error) {
            console.error('Error fetching menu:', error);
            
            // If we have cached data, use it
            if (this.menuData) {
                this.showError('Connection issue. Displaying cached menu.');
            } else {
                this.showError('Unable to load menu. Please check your connection.');
            }
        }
    }
    
    renderMenu(data) {
        // Hide loading spinner
        if (this.loading) {
            this.loading.style.display = 'none';
        }
        
        // Update business name
        this.businessName.textContent = data.board.businessName;
        
        // Apply theme
        this.applyTheme(data.board.theme);
        
        // Clear existing content
        this.container.innerHTML = '';
        
        // Render categories
        data.categories.forEach(category => {
            if (category.items && category.items.length > 0) {
                const categoryElement = this.createCategoryElement(category);
                this.container.appendChild(categoryElement);
            }
        });
        
        // Update last updated time
        const updateTime = new Date(data.lastUpdated);
        this.lastUpdated.textContent = `Last updated: ${updateTime.toLocaleTimeString()}`;
        
        // Add update animation
        this.container.classList.add('updating');
        setTimeout(() => {
            this.container.classList.remove('updating');
        }, 500);
    }
    
    createCategoryElement(category) {
        const categoryDiv = document.createElement('div');
        categoryDiv.className = 'menu-category';
        
        const categoryHeader = document.createElement('h2');
        categoryHeader.className = 'category-header';
        categoryHeader.textContent = category.name;
        categoryDiv.appendChild(categoryHeader);
        
        const itemsGrid = document.createElement('div');
        itemsGrid.className = 'menu-items-grid';
        
        category.items.forEach(item => {
            const itemElement = this.createItemElement(item);
            itemsGrid.appendChild(itemElement);
        });
        
        categoryDiv.appendChild(itemsGrid);
        return categoryDiv;
    }
    
    createItemElement(item) {
        const itemDiv = document.createElement('div');
        itemDiv.className = 'menu-item';
        
        if (item.isFeatured) {
            itemDiv.classList.add('featured');
        }
        
        if (!item.isAvailable) {
            itemDiv.classList.add('unavailable');
        }
        
        // Image
        if (item.imageUrl) {
            const img = document.createElement('img');
            img.src = item.imageUrl;
            img.alt = item.name;
            img.className = 'item-image';
            img.onerror = () => {
                img.style.display = 'none';
                const placeholder = this.createPlaceholderImage();
                itemDiv.insertBefore(placeholder, itemDiv.firstChild);
            };
            itemDiv.appendChild(img);
        } else {
            const placeholder = this.createPlaceholderImage();
            itemDiv.appendChild(placeholder);
        }
        
        // Details
        const detailsDiv = document.createElement('div');
        detailsDiv.className = 'item-details';
        
        const nameDiv = document.createElement('div');
        nameDiv.className = 'item-name';
        nameDiv.textContent = item.name;
        detailsDiv.appendChild(nameDiv);
        
        if (item.description) {
            const descDiv = document.createElement('div');
            descDiv.className = 'item-description';
            descDiv.textContent = item.description;
            detailsDiv.appendChild(descDiv);
        }
        
        // Price container
        const priceContainer = document.createElement('div');
        priceContainer.className = 'item-price-container';
        
        const priceDiv = document.createElement('div');
        priceDiv.className = 'item-price';
        priceDiv.textContent = `$${item.price}`;
        priceContainer.appendChild(priceDiv);
        
        // Show original price if scheduled price is active
        if (item.originalPrice && item.activeSchedule) {
            const originalPriceDiv = document.createElement('div');
            originalPriceDiv.className = 'original-price';
            originalPriceDiv.textContent = `$${item.originalPrice}`;
            priceContainer.appendChild(originalPriceDiv);
            
            const scheduleBadge = document.createElement('div');
            scheduleBadge.className = 'schedule-badge';
            scheduleBadge.textContent = item.activeSchedule;
            priceContainer.appendChild(scheduleBadge);
        }
        
        detailsDiv.appendChild(priceContainer);
        itemDiv.appendChild(detailsDiv);
        
        // Badges
        if (item.isFeatured) {
            const featuredBadge = document.createElement('div');
            featuredBadge.className = 'featured-badge';
            featuredBadge.textContent = '⭐ Featured';
            itemDiv.appendChild(featuredBadge);
        }
        
        if (!item.isAvailable) {
            const unavailableBadge = document.createElement('div');
            unavailableBadge.className = 'unavailable-badge';
            unavailableBadge.textContent = 'Sold Out';
            itemDiv.appendChild(unavailableBadge);
        }
        
        return itemDiv;
    }
    
    createPlaceholderImage() {
        const placeholder = document.createElement('div');
        placeholder.className = 'item-image placeholder';
        placeholder.textContent = '🍽️';
        return placeholder;
    }
    
    applyTheme(theme) {
        if (!theme) return;
        
        const root = document.documentElement;
        
        if (theme.primaryColor) {
            root.style.setProperty('--primary-color', theme.primaryColor);
        }
        if (theme.secondaryColor) {
            root.style.setProperty('--secondary-color', theme.secondaryColor);
        }
        if (theme.bgColor) {
            document.body.style.background = theme.bgColor;
        }
        if (theme.fontFamily) {
            document.body.style.fontFamily = `'${theme.fontFamily}', sans-serif`;
        }
    }
    
    saveToCache(data) {
        try {
            localStorage.setItem(this.cacheKey, JSON.stringify(data));
            localStorage.setItem(`${this.cacheKey}_timestamp`, Date.now().toString());
        } catch (error) {
            console.error('Error saving to cache:', error);
        }
    }
    
    loadFromCache() {
        try {
            const cachedData = localStorage.getItem(this.cacheKey);
            if (cachedData) {
                this.menuData = JSON.parse(cachedData);
                this.renderMenu(this.menuData);
                console.log('Loaded menu from cache');
            }
        } catch (error) {
            console.error('Error loading from cache:', error);
        }
    }
    
    startAutoRefresh() {
        if (this.refreshTimer) {
            clearInterval(this.refreshTimer);
        }
        
        this.refreshTimer = setInterval(() => {
            if (this.isOnline) {
                this.fetchMenu();
            }
        }, this.refreshInterval);
    }
    
    updateRefreshInterval(newInterval) {
        if (newInterval !== this.refreshInterval) {
            this.refreshInterval = newInterval;
            this.startAutoRefresh();
            console.log(`Refresh interval updated to ${newInterval}ms`);
        }
    }
    
    showError(message) {
        this.container.innerHTML = `
            <div class="error-message">
                <p>⚠️ ${message}</p>
            </div>
        `;
    }
    
    destroy() {
        if (this.refreshTimer) {
            clearInterval(this.refreshTimer);
        }
    }
}

// Service Worker registration for offline support
if ('serviceWorker' in navigator) {
    window.addEventListener('load', () => {
        navigator.serviceWorker.register('../sw.js')
            .then(registration => {
                console.log('ServiceWorker registered:', registration);
            })
            .catch(error => {
                console.log('ServiceWorker registration failed:', error);
            });
    });
}
