class MenuAdmin {
    constructor() {
        this.apiUrl = '../api';
        this.updateApiUrl = '../api/update-menu.php';
        this.getMenuUrl = '../api/get-menu.php';
        this.apiKey = 'your_secret_api_key_change_this'; // Must match config.php
        this.boardSlug = 'demo-main-board';
        this.menuData = null;
        this.categories = [];
        
        this.initElements();
        this.initEventListeners();
        this.loadMenu();
    }
    
    initElements() {
        // Navigation
        this.navItems = document.querySelectorAll('.nav-item');
        this.sections = document.querySelectorAll('.content-section');
        
        // Buttons
        this.addItemBtn = document.getElementById('add-item-btn');
        this.refreshBtn = document.getElementById('refresh-btn');
        
        // Modal
        this.modal = document.getElementById('item-modal');
        this.modalTitle = document.getElementById('modal-title');
        this.closeModal = document.getElementById('close-modal');
        this.cancelBtn = document.getElementById('cancel-btn');
        this.itemForm = document.getElementById('item-form');
        
        // Form fields
        this.itemId = document.getElementById('item-id');
        this.itemCategory = document.getElementById('item-category');
        this.itemName = document.getElementById('item-name');
        this.itemDescription = document.getElementById('item-description');
        this.itemPrice = document.getElementById('item-price');
        this.itemImage = document.getElementById('item-image');
        this.itemAvailable = document.getElementById('item-available');
        this.itemFeatured = document.getElementById('item-featured');
        
        // Containers
        this.menuItemsContainer = document.getElementById('menu-items-container');
        this.categoryFilter = document.getElementById('category-filter');
        this.searchInput = document.getElementById('search-items');
    }
    
    initEventListeners() {
        // Navigation
        this.navItems.forEach(item => {
            item.addEventListener('click', (e) => {
                e.preventDefault();
                const section = item.dataset.section;
                this.switchSection(section);
            });
        });
        
        // Buttons
        this.addItemBtn.addEventListener('click', () => this.openAddItemModal());
        this.refreshBtn.addEventListener('click', () => this.loadMenu());
        
        // Modal
        this.closeModal.addEventListener('click', () => this.closeItemModal());
        this.cancelBtn.addEventListener('click', () => this.closeItemModal());
        this.modal.addEventListener('click', (e) => {
            if (e.target === this.modal) {
                this.closeItemModal();
            }
        });
        
        // Form
        this.itemForm.addEventListener('submit', (e) => {
            e.preventDefault();
            this.saveItem();
        });
        
        // Filters
        this.categoryFilter.addEventListener('change', () => this.filterItems());
        this.searchInput.addEventListener('input', () => this.filterItems());
    }
    
    switchSection(sectionName) {
        this.navItems.forEach(item => item.classList.remove('active'));
        this.sections.forEach(section => section.classList.remove('active'));
        
        const activeNav = document.querySelector(`[data-section="${sectionName}"]`);
        const activeSection = document.getElementById(`${sectionName}-section`);
        
        if (activeNav) activeNav.classList.add('active');
        if (activeSection) activeSection.classList.add('active');
    }
    
    async loadMenu() {
        this.showLoading();
        
        try {
            const response = await fetch(`${this.getMenuUrl}?board=${this.boardSlug}`);
            const data = await response.json();
            
            if (data.success) {
                this.menuData = data;
                this.categories = data.categories;
                this.populateCategoryFilters();
                this.renderMenuItems();
            } else {
                this.showError(data.error || 'Failed to load menu');
            }
        } catch (error) {
            console.error('Error loading menu:', error);
            this.showError('Failed to load menu. Please check your connection.');
        }
    }
    
    populateCategoryFilters() {
        // Clear existing options
        this.itemCategory.innerHTML = '<option value="">Select Category</option>';
        this.categoryFilter.innerHTML = '<option value="">All Categories</option>';
        
        this.categories.forEach(category => {
            // For item form
            const option1 = document.createElement('option');
            option1.value = category.id;
            option1.textContent = category.name;
            this.itemCategory.appendChild(option1);
            
            // For filter
            const option2 = document.createElement('option');
            option2.value = category.id;
            option2.textContent = category.name;
            this.categoryFilter.appendChild(option2);
        });
    }
    
    renderMenuItems() {
        this.menuItemsContainer.innerHTML = '';
        
        const allItems = [];
        this.categories.forEach(category => {
            if (category.items) {
                category.items.forEach(item => {
                    allItems.push({
                        ...item,
                        categoryName: category.name,
                        categoryId: category.id
                    });
                });
            }
        });
        
        if (allItems.length === 0) {
            this.menuItemsContainer.innerHTML = '<p class="text-muted">No menu items found.</p>';
            return;
        }
        
        allItems.forEach(item => {
            const itemCard = this.createItemCard(item);
            this.menuItemsContainer.appendChild(itemCard);
        });
    }
    
    createItemCard(item) {
        const card = document.createElement('div');
        card.className = 'item-card';
        card.dataset.itemId = item.id;
        card.dataset.categoryId = item.categoryId;
        
        const badges = [];
        if (item.isFeatured) badges.push('<span class="badge badge-featured">Featured</span>');
        if (item.isAvailable) {
            badges.push('<span class="badge badge-available">Available</span>');
        } else {
            badges.push('<span class="badge badge-unavailable">Unavailable</span>');
        }
        
        card.innerHTML = `
            <div class="item-card-header">
                <div>
                    <div class="item-title">${this.escapeHtml(item.name)}</div>
                    <div class="item-category">${this.escapeHtml(item.categoryName)}</div>
                </div>
                <div class="item-actions">
                    <button class="btn-icon" onclick="menuAdmin.editItem(${item.id})" title="Edit">
                        ✏️
                    </button>
                    <button class="btn-icon" onclick="menuAdmin.deleteItem(${item.id})" title="Delete">
                        🗑️
                    </button>
                </div>
            </div>
            <div class="item-description">
                ${this.escapeHtml(item.description || 'No description')}
            </div>
            <div class="item-footer">
                <div class="item-price">$${item.price}</div>
                <div class="item-badges">
                    ${badges.join('')}
                </div>
            </div>
        `;
        
        return card;
    }
    
    filterItems() {
        const categoryId = this.categoryFilter.value;
        const searchTerm = this.searchInput.value.toLowerCase();
        
        const cards = this.menuItemsContainer.querySelectorAll('.item-card');
        
        cards.forEach(card => {
            const itemCategoryId = card.dataset.categoryId;
            const itemName = card.querySelector('.item-title').textContent.toLowerCase();
            const itemDescription = card.querySelector('.item-description').textContent.toLowerCase();
            
            const matchesCategory = !categoryId || itemCategoryId === categoryId;
            const matchesSearch = !searchTerm || 
                itemName.includes(searchTerm) || 
                itemDescription.includes(searchTerm);
            
            if (matchesCategory && matchesSearch) {
                card.style.display = 'block';
            } else {
                card.style.display = 'none';
            }
        });
    }
    
    openAddItemModal() {
        this.modalTitle.textContent = 'Add Menu Item';
        this.itemForm.reset();
        this.itemId.value = '';
        this.itemAvailable.checked = true;
        this.modal.classList.add('active');
    }
    
    editItem(itemId) {
        // Find the item
        let item = null;
        let categoryId = null;
        
        for (const category of this.categories) {
            const foundItem = category.items.find(i => i.id === itemId);
            if (foundItem) {
                item = foundItem;
                categoryId = category.id;
                break;
            }
        }
        
        if (!item) {
            alert('Item not found');
            return;
        }
        
        // Populate form
        this.modalTitle.textContent = 'Edit Menu Item';
        this.itemId.value = item.id;
        this.itemCategory.value = categoryId;
        this.itemName.value = item.name;
        this.itemDescription.value = item.description || '';
        this.itemPrice.value = item.price;
        this.itemImage.value = item.imageUrl || '';
        this.itemAvailable.checked = item.isAvailable;
        this.itemFeatured.checked = item.isFeatured;
        
        this.modal.classList.add('active');
    }
    
    async deleteItem(itemId) {
        if (!confirm('Are you sure you want to delete this item?')) {
            return;
        }
        
        try {
            const response = await fetch(this.updateApiUrl, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.apiKey}`
                },
                body: JSON.stringify({ item_id: itemId })
            });
            
            const result = await response.json();
            
            if (result.success) {
                alert('Item deleted successfully!');
                this.loadMenu();
            } else {
                alert('Error: ' + (result.error || 'Failed to delete item'));
            }
        } catch (error) {
            console.error('Error deleting item:', error);
            alert('Failed to delete item. Please try again.');
        }
    }
    
    async saveItem() {
        const itemId = this.itemId.value;
        const isEdit = itemId !== '';
        
        const itemData = {
            category_id: parseInt(this.itemCategory.value),
            item_name: this.itemName.value,
            description: this.itemDescription.value,
            price: parseFloat(this.itemPrice.value),
            image_url: this.itemImage.value || null,
            is_available: this.itemAvailable.checked,
            is_featured: this.itemFeatured.checked
        };
        
        if (isEdit) {
            itemData.item_id = parseInt(itemId);
        }
        
        try {
            const response = await fetch(this.updateApiUrl, {
                method: isEdit ? 'PUT' : 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.apiKey}`
                },
                body: JSON.stringify(itemData)
            });
            
            const result = await response.json();
            
            if (result.success) {
                alert(isEdit ? 'Item updated successfully!' : 'Item added successfully!');
                this.closeItemModal();
                this.loadMenu();
            } else {
                alert('Error: ' + (result.error || 'Failed to save item'));
            }
        } catch (error) {
            console.error('Error saving item:', error);
            alert('Failed to save item. Please try again.');
        }
    }
    
    closeItemModal() {
        this.modal.classList.remove('active');
        this.itemForm.reset();
    }
    
    showLoading() {
        this.menuItemsContainer.innerHTML = `
            <div class="loading-state">
                <div class="spinner"></div>
                <p>Loading menu items...</p>
            </div>
        `;
    }
    
    showError(message) {
        this.menuItemsContainer.innerHTML = `
            <div class="error-message">
                <p>⚠️ ${message}</p>
            </div>
        `;
    }
    
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
}

// Initialize admin when DOM is loaded
let menuAdmin;
document.addEventListener('DOMContentLoaded', () => {
    menuAdmin = new MenuAdmin();
});
