<?php
require_once 'config.php';

// Get menu board by slug
$boardSlug = isset($_GET['board']) ? $_GET['board'] : null;

if (!$boardSlug) {
    http_response_code(400);
    echo json_encode(['error' => 'Board slug is required']);
    exit;
}

try {
    $conn = getDBConnection();
    
    // Get board details
    $stmt = $conn->prepare("
        SELECT mb.*, u.business_name 
        FROM menu_boards mb
        JOIN users u ON mb.user_id = u.id
        WHERE mb.board_slug = ? AND mb.is_active = 1
    ");
    $stmt->execute([$boardSlug]);
    $board = $stmt->fetch();
    
    if (!$board) {
        http_response_code(404);
        echo json_encode(['error' => 'Board not found']);
        exit;
    }
    
    // Get categories with items
    $stmt = $conn->prepare("
        SELECT 
            mc.id as category_id,
            mc.category_name,
            mc.display_order as category_order,
            mi.id as item_id,
            mi.item_name,
            mi.description,
            mi.price,
            mi.image_url,
            mi.is_available,
            mi.is_featured,
            mi.display_order as item_order
        FROM menu_categories mc
        LEFT JOIN menu_items mi ON mc.id = mi.category_id
        WHERE mc.board_id = ? AND mc.is_visible = 1
        ORDER BY mc.display_order, mi.display_order
    ");
    $stmt->execute([$board['id']]);
    $results = $stmt->fetchAll();
    
    // Get active price schedules
    $stmt = $conn->prepare("
        SELECT 
            ps.item_id,
            ps.scheduled_price,
            ps.start_time,
            ps.end_time,
            ps.days_of_week,
            ps.schedule_name
        FROM price_schedules ps
        WHERE ps.is_active = 1 
        AND ps.item_id IN (
            SELECT mi.id 
            FROM menu_items mi 
            JOIN menu_categories mc ON mi.category_id = mc.id 
            WHERE mc.board_id = ?
        )
    ");
    $stmt->execute([$board['id']]);
    $schedules = $stmt->fetchAll();
    
    // Organize schedules by item_id
    $schedulesMap = [];
    foreach ($schedules as $schedule) {
        $itemId = $schedule['item_id'];
        if (!isset($schedulesMap[$itemId])) {
            $schedulesMap[$itemId] = [];
        }
        $schedulesMap[$itemId][] = $schedule;
    }
    
    // Check if current time matches any schedule
    $currentDay = strtolower(date('l'));
    $currentTime = date('H:i:s');
    
    function isScheduleActive($schedule, $currentDay, $currentTime) {
        $days = json_decode($schedule['days_of_week'], true);
        if (!in_array($currentDay, $days)) {
            return false;
        }
        return ($currentTime >= $schedule['start_time'] && $currentTime <= $schedule['end_time']);
    }
    
    // Organize data by categories
    $categories = [];
    $currentCategory = null;
    
    foreach ($results as $row) {
        $categoryId = $row['category_id'];
        
        if ($currentCategory === null || $currentCategory['id'] !== $categoryId) {
            if ($currentCategory !== null) {
                $categories[] = $currentCategory;
            }
            $currentCategory = [
                'id' => $categoryId,
                'name' => $row['category_name'],
                'items' => []
            ];
        }
        
        if ($row['item_id']) {
            $itemId = $row['item_id'];
            $price = $row['price'];
            $activeSchedule = null;
            
            // Check for active scheduled price
            if (isset($schedulesMap[$itemId])) {
                foreach ($schedulesMap[$itemId] as $schedule) {
                    if (isScheduleActive($schedule, $currentDay, $currentTime)) {
                        $price = $schedule['scheduled_price'];
                        $activeSchedule = $schedule['schedule_name'];
                        break;
                    }
                }
            }
            
            $currentCategory['items'][] = [
                'id' => $itemId,
                'name' => $row['item_name'],
                'description' => $row['description'],
                'price' => number_format($price, 2),
                'originalPrice' => $activeSchedule ? number_format($row['price'], 2) : null,
                'imageUrl' => $row['image_url'],
                'isAvailable' => (bool)$row['is_available'],
                'isFeatured' => (bool)$row['is_featured'],
                'activeSchedule' => $activeSchedule
            ];
        }
    }
    
    if ($currentCategory !== null) {
        $categories[] = $currentCategory;
    }
    
    // Parse theme config
    $themeConfig = json_decode($board['theme_config'], true);
    
    // Build response
    $response = [
        'success' => true,
        'board' => [
            'id' => $board['id'],
            'name' => $board['board_name'],
            'businessName' => $board['business_name'],
            'refreshInterval' => $board['refresh_interval'],
            'theme' => $themeConfig
        ],
        'categories' => $categories,
        'lastUpdated' => date('c')
    ];
    
    echo json_encode($response);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
}
?>
